defmodule Partie do
  @moduledoc """
  Serveur générique contenant une partie de jeu de cartes en cours.
  """

  use GenServer

  @enforce_keys [:id, :règles, :jeu]
  defstruct [:id, :règles, :jeu]

  @type t() :: %{
          id: integer(),
          règles: module(),
          jeu: struct()
        }

  @doc """
  Crée une nouvelle partie.
  """
  @spec start_link({integer(), module()}, GenServer.options()) :: GenServer.on_start()
  def start_link({id, règles}, options \\ []) do
    options = Keyword.put(options, :name, nom_processus(id))
    GenServer.start_link(__MODULE__, {id, règles}, options)
  end

  @doc """
  Demande la version condensée du jeu pour le joueur spécifié.

  Le PID du serveur (ou l'identifiant de la partie) ainsi que
  l'identifiant du joueur (un entier) sont à préciser en paramètre.
  """
  @spec condensée(integer() | pid(), integer()) :: map()
  def condensée(id, joueur) when is_integer(id) do
    condensée(nom_processus(id), joueur)
  end

  def condensée(processus, joueur) do
    GenServer.call(processus, {:condensée, joueur})
  end

  @doc """
  Tente d'ajouter un joueur.

  Si le jeu refuse d'ajouter le joueur, l'atome :invalide est retourné.
  Sinon, le joueur est ajouté à la partie en cours et le jeu condensé
  est retourné.
  """
  @spec ajouter_joueur(integer() | pid(), String.t()) :: struct() | :invalide
  def ajouter_joueur(id, nom) when is_integer(id) do
    ajouter_joueur(nom_processus(id), nom)
  end

  def ajouter_joueur(pid, nom) do
    GenServer.call(pid, {:ajouter_joueur, nom})
  end

  @doc """
  Retourne une liste de coups utilisables par le joueur spécifié.

  Chaque coup est représenté par un tuple : l'identifiant du coup (tel qu'à
  envoyer à `jouer`), le nom du coup et un booléen indiquant si
  ce coup est valide pour ce joueur.
  """
  @spec coups(integer() | pid(), integer()) :: [{atom() | tuple(), String.t(), boolean()}]
  def coups(id, joueur) when is_integer(id) do
    coups(nom_processus(id), joueur)
  end

  def coups(pid, joueur) do
    GenServer.call(pid, {:coups, joueur})
  end

  @doc """
  Fait jouer le joueur précisé.

  Le coup est un atome ou un tuple dont le premier élément est un atome.
  Si le coup est accepté, retourne le jeu, sinon retourne :invalide.
  """
  @spec jouer(integer() | pid(), integer(), atom() | any()) :: struct() | :invalide
  def jouer(id, joueur, coup) when is_integer(id) do
    jouer(nom_processus(id), joueur, coup)
  end

  def jouer(pid, joueur, coup) do
    GenServer.call(pid, {:jouer, joueur, coup})
  end

  @impl true
  @spec init(module()) :: {:ok, t()}
  def init({id, règles}) do
    {:ok, %Partie{id: id, règles: règles, jeu: règles.new()}}
  end

  @impl true
  def handle_call({:condensée, joueur}, _from, partie) do
    {:reply, condenser_pour(partie, joueur), partie}
  end

  @impl true
  def handle_call({:ajouter_joueur, nom}, _from, partie) do
    if partie.règles.ajouter_joueur?(partie.jeu) do
      {jeu, joueur} = partie.règles.ajouter_joueur(partie.jeu, nom)
      partie = %{partie | jeu: jeu}
      {:reply, condenser_pour(partie, joueur), partie}
    else
      {:reply, :invalide, partie}
    end
  end

  @impl true
  def handle_call({:coups, joueur}, _from, partie) do
    coups = partie.règles.coups(partie.jeu, joueur)
    {:reply, coups, partie}
  end

  @impl true
  def handle_call({:jouer, joueur, coup}, _from, partie) do
    if partie.règles.jouer?(partie.jeu, joueur, coup) do
      jeu = partie.règles.jouer(partie.jeu, joueur, coup)
      partie = %{partie | jeu: jeu}
      {:reply, condenser_pour(partie, joueur), partie}
    else
      {:reply, :invalide, partie}
    end
  end

  @spec condenser_pour(t(), integer()) :: map()
  defp condenser_pour(partie, joueur) do
    partie.règles.condensée(partie.jeu, joueur)
  end

  @spec nom_processus(integer()) :: tuple()
  defp nom_processus(id), do: {:via, Registry, {Partie.Registre, id}}
end
